#pragma once

#include <algorithm>
#include <functional>
#include <initializer_list>
#include <numeric>

namespace fpp {

    template<typename Ct> struct container;

    namespace utility {

        template<typename Ct>
        container<Ct>& concat_in_place(container<Ct>& target) {
            return target;
        }

        template<typename Ct, typename... Args>
        container<Ct>& concat_in_place(container<Ct>& target, const Ct& source, Args... sources) {
            for (auto&& i : source) {
                target.stlc.push_back(i);
            }
            concat_in_place(target, sources...);
            return target;
        }

        template<typename Ct, typename... Args>
        container<Ct>& concat_in_place(container<Ct>& target, const container<Ct>& source, Args... sources) {
            concat_in_place(target, source.stlc, sources...);
            return target;
        }


        template<typename Vt, typename Rt>
        using index_callback = std::function<Rt(Vt, size_t)>;

        template<typename Vt, typename Rt>
        using value_callback = std::function<Rt(Vt)>;

        template<typename Vt, typename Rt>
        using index_reduce_op = std::function<Rt(Rt, Vt, size_t)>;

        template<typename Vt, typename Rt>
        using value_reduce_op = std::function<Rt(Rt, Vt)>;


        template<typename Vt, typename Rt>
        inline index_callback<Vt, Rt> to_index_callback(value_callback<Vt, Rt>& fn) {
            return [&fn](Vt v, size_t) -> Rt {
                return fn(v);
            };
        }

    };

    template<typename Ct> struct container {

        // member types
        typedef typename Ct::value_type value_type;
        typedef typename Ct::allocator_type allocator_type;
        typedef typename Ct::size_type size_type;
        typedef typename Ct::difference_type difference_type;
        typedef typename Ct::reference reference;
        typedef typename Ct::const_reference const_reference;
        typedef typename Ct::pointer pointer;
        typedef typename Ct::const_pointer const_pointer;
        typedef typename Ct::iterator iterator;
        typedef typename Ct::const_iterator const_iterator;
        typedef typename Ct::reverse_iterator reverse_iterator;
        typedef typename Ct::const_reverse_iterator const_reverse_iterator;

        typedef utility::value_callback<value_type, bool> value_predicate;
        typedef utility::index_callback<value_type, bool> index_predicate;

        typedef utility::value_callback<value_type, void> value_procedure;
        typedef utility::index_callback<value_type, void> index_procedure;


        // member fields
        Ct stlc;


        // constructors
        container() : stlc{} {}
        container(const Ct& cpp) : stlc{ cpp } {}
        container(Ct&& cpp) : stlc{ std::move(cpp) } {}
        container(const container<Ct>& that) : stlc{ that.stlc } {}
        container(container<Ct>&& that) : stlc{ std::move(that.stlc) } {}
        container(const std::initializer_list<value_type>& list) : stlc{ list } {}


        // iterators
        iterator begin() {
            return stlc.begin();
        }

        iterator end() {
            return stlc.end();
        }

        template<typename... Ts>
        container<Ct> concat(Ts... these) {
            container<Ct> ret{};
            utility::concat_in_place(ret, stlc, these...);
            return ret;
        }


        // every
        bool every(index_predicate fn) {
            size_t i{};
            return std::all_of(stlc.begin(), stlc.end(), [&i](value_type v) -> bool {
                auto ret = fn(v, i);
                i++;
                return ret;
            });
        }

        inline bool every(value_predicate fn) {
            return std::all_of(stlc.begin(), stlc.end(), fn);
        }


        // filter
        container<Ct> filter(index_predicate fn) {
            container<Ct> ret{};
            size_t i{};
            for (auto& v : stlc) {
                if (fn(v, i)) {
                    ret.stlc.push_back(v);
                }
                i++;
            }
            return ret;
        }

        inline container<Ct> filter(value_predicate fn) {
            return filter(utility::to_index_callback(fn));
        }


        // find
        iterator find(index_predicate fn) {
            size_t i{};
            return std::find_if(stlc.begin(), stlc.end(), [&fn, &i](value_type v) -> bool {
                auto ret = fn(v, i);
                i++;
                return ret;
            });
        }

        inline iterator find(value_predicate fn) {
            return std::find_if(stlc.begin(), stlc.end(), fn);
        }

        
        // find_index
        size_t find_index(index_predicate fn) {
            size_t i{};
            std::find_if(stlc.begin(), stlc.end(), [&fn, &i](value_type v) -> bool {
                auto ret = fn(v, i);
                i++;
                return ret;
            });

            return i == stlc.size() ? -1 : i - 1;
        }

        inline size_t find_index(value_predicate fn) {
            return find_index(utility::to_index_callback(fn));
        }


        // for_each
        void for_each(index_procedure fn) {
            size_t i{};
            for (auto& v : stlc) {
                fn(v, i);
                i++;
            }
        }

        inline size_t for_each(value_procedure fn) {
            return for_each(utility::to_index_callback(fn));
        }


        // includes
        inline bool includes(value_type v) {
            return std::find(stlc.begin(), stlc.end(), v) != stlc.end();
        }


        // index_of
        inline size_t index_of(value_type val) {
            return find_index([&val](value_type v) -> bool {
                return v == val;
            });
        }


        // last_index_of
        size_t last_index_of(value_type val) {
            size_t i{ stlc.size() };
            auto it = std::find_if(stlc.rbegin(), stlc.rend(),
                [&val, &i](value_type v) -> bool {
                i--;
                return v == val;
            });
            return it == stlc.rend() ? -1 : i;
        }


        // map
        template<typename Rct = Ct>
        container<Rct> map(utility::index_callback<value_type, typename Rct::value_type> fn) {
            container<Rct> ret{};
            ret.stlc.reserve(stlc.size());

            size_t i{};
            std::transform(stlc.begin(), stlc.end(), std::back_inserter(ret.stlc),
                [&fn, &i](value_type v) -> typename Rct::value_type {
                return fn(v, i++);
            });
            return ret;
        }

        template<typename Rct = Ct>
        inline container<Rct>
            map(utility::value_callback<value_type, typename Rct::value_type> fn) {

            return map<Rct>(utility::to_index_callback<
                value_type, typename Rct::value_type
            >(fn));
        }


        // reduce
        template<typename Rt = value_type>
        Rt reduce(utility::index_reduce_op<value_type, typename Rt> fn, Rt init = Rt{}) {
            size_t i{};
            return std::accumulate(stlc.begin(), stlc.end(), init,
                [&fn, &i](Rt acc, value_type v) {
                return fn(acc, v, i++);
            });
        }

        template<typename Rt = value_type>
        inline Rt reduce(utility::value_reduce_op<value_type, typename Rt> fn, Rt init = Rt{}) {
            return std::accumulate(stlc.begin(), stlc.end(), init, fn);
        }

    };

}
